; Subclass.asm - demonstrates window subclassing
;
; After using child window controls it quickly becomes apparent
; that whilst they are extremely useful and save us a huge amount
; of work, they lack one very important feature - their behaviour
; is always the same and uncontrollable.
;
; Perhaps this needs clarifying. Consider an edit control in which
; you only want the user to be able to enter certain types of
; string, say hexadecimal values only. If you have an 'Ok' button
; or something you could examine the string in the edit control
; when it's clicked, but this is hardly an ideal solution.
;
; Clearly, the ideal way of providing this behaviour would be if
; we could process WM_CHAR messages, but how can we?
;
; Pause for a moment to remember how child window controls differ
; from normal child windows - the main thing is that we don't need
; to register a window class as Windows already predefines them.
; If you think about it, normally we'd register a window class
; with our own window procedure and use the window procedure to
; handle all messages sent to the window, including WM_CHAR.
; However, with a child window control this window procedure is
; hidden away somewhere in Windows.
;
; The obvious solution would seem to be if we could somehow get
; access to this procedure and use it to handle all messages
; except those that we want to handle ourselves.
;
; Fortunately, we're in luck and Windows provides for doing this
; very thing - it's called window subclassing. Note that it's
; not _control_ subclassing but _window_ as it's any _window_
; that can be subclassed.
;
; So, how do we actually implement this in practice?
;
; First thing to do is to set the new child window control's
; window procedure, which can be done by calling SetWindowLong.
; Hang on though, don't we need to get the previous address
; before setting the new one? Well no, because SetWindowLong
; actually returns this for us when we set the new one, which
; is quite handy.
;
; And after all that, we setup the new window procedure to
; intercept the messages we want and use CallWindowProc when
; we want the original window procedure to process it.
;
; In this example, I'll demonstrate making an edit control
; only accept hexadecimal numbers.
;

%define _WINMESSAGES_
%define _WINVKEYS_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

ddglobal _gbl_hInstance

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	WinMainPrologue
	mov	eax, .hInstance
	mov	._gbl_hInstance, eax
	mov	esi, ._wndclass
	mov	edi, ._msg
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	sc RegisterClassEx, esi
	cmp	eax, TRUE
	je	near _WinMain_Fail
	TEXTlocal szWndCaption, 'Subclassing an edit control',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	mov	._hwnd, eax
	sc ShowWindow, ._hwnd, .nCmdShow
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	sc GetMessage, ._msg, NULL, 0, 0
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	sc TranslateMessage, ._msg
	sc DispatchMessage, ._msg
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
proc _WndProc, hwnd, message, wParam, lParam
	ddglobal	_hwndEdit
	ddglobal	_hwndProc
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_CREATE
			sc CreateWindowEx, 0, _szClassNameEdit, 0, WS_CHILDWINDOW | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL | ES_MULTILINE | ES_NOHIDESEL | ES_WANTRETURN, 0, 0, 0, 0, .hwnd, 1, ._gbl_hInstance, NULL
			mov	._hwndEdit, eax
			;
			; Having created the edit control, we need to set the new
			; window procedure it will use via SetWindowLong which
			; takes three parameters:
			;
			;   1. Window handle
			;   2. Offset of value to set
			;   3. New value
			;
			; The offset can be one of several predefined constants,
			; see the help for a complete list, but the one we're
			; interested in is GWL_WNDPROC.
			;
			sc SetWindowLong, eax, GWL_WNDPROC, _SubclassProc
			;
			; And save the old window procedure address
			;
			mov	._hwndProc, eax
			xor	eax,eax
			break
		case WM_SIZE
			;
			; Resize, so make the edit control fill the whole window
			;
			mov	eax, .lParam
			movsx	esi, ax
			shr	eax, 16
			movsx	edi, ax
			sc MoveWindow, ._hwndEdit, 0, 0, esi, edi, TRUE
			xor	eax, eax
			break
		case WM_DESTROY
			sc PostQuitMessage, 0
			xor	eax,eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
; The subclassed window procedure.
;
; This works just like a normal window procedure, ie it is a Callback
; and we process any messages receieved, and / or pass the message
; onto the original window procedure. 
;
proc _SubclassProc, hwnd, message, wParam, lParam
	;
	CallbackPrologue
	;
	switch .message
		case WM_CHAR
			;
			; WM_CHAR message?
			;
			; Yes, so get the character in al (it's held in wParam)
			;
			mov	eax, .wParam
			;
			; ecx holds whether it's a valid character (zero = no)
			;
			xor	ecx, ecx
			;
			; check if it's backspace, 0-9, A-F or a-f
			;
			if al, e, VK_BACK
				mov	ecx, 1
			else
				ifand al, ge, '0', al, le, '9'
					mov	ecx, 1
				else
					ifand al, ge, 'A', al, le, 'F'
						mov	ecx, 1
					else
						ifand al, ge, 'a', al, le, 'f'
							mov	ecx, 1
						endif
					endif
				endif
			endif
			if ecx, e, 0
				;
				; invalid character
				;
				xor	eax, eax
			else
				;
				; valid, so pass the message onto the original window procedure
				;
				sc CallWindowProc, ._hwndProc, .hwnd, WM_CHAR, eax, .lParam
			endif
			break
		default
			;
			; It's not a WM_CHAR message, so use CallWindowProc to
			; execute the old window procedure, which takes 5 parameters:
			;
			;   1. Old window procedure address
			;   2. Window handle
			;   3. Message to process
			;   4. wParam of message
			;   5. lParam of message
			;
			sc CallWindowProc, ._hwndProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
[section .bss]

[section .data]
